<?php
/*
Plugin Name: Partner Auto Traffic Ranker Daily Reset
Description: Public partner signup with manual approval. Use shortcode [partner_ranking] or widget to show top sites sending visits daily (daily count resets at midnight). Admin can configure how many partners to show. Settings > Traffic Partners.
Version: 2.0
Author: Alexandre Lavrador
Text Domain: partner-traffic-ranker
Domain Path: /languages
*/


if ( ! defined( 'ABSPATH' ) ) exit;

add_action( 'plugins_loaded', 'patr_load_textdomain' );
function patr_load_textdomain() {
    load_plugin_textdomain( 'partner-traffic-ranker', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
}

register_activation_hook( __FILE__, 'patr_create_tables' );
function patr_create_tables() {
    global $wpdb;
    $charset = $wpdb->get_charset_collate();

    $table_partners = $wpdb->prefix . 'partner_traffic';
    $table_clicks = $wpdb->prefix . 'partner_daily_clicks';
    $table_ip_log = $wpdb->prefix . 'partner_click_ips';

    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

    dbDelta("CREATE TABLE $table_partners (
        id INT AUTO_INCREMENT PRIMARY KEY,
        partner_domain VARCHAR(255) NOT NULL UNIQUE,
        status ENUM('pending','approved','rejected') DEFAULT 'pending'
    ) $charset;");

    dbDelta("CREATE TABLE $table_clicks (
        id INT AUTO_INCREMENT PRIMARY KEY,
        partner_id INT NOT NULL,
        click_date DATE NOT NULL,
        clicks INT DEFAULT 0,
        UNIQUE KEY partner_date (partner_id, click_date),
        FOREIGN KEY (partner_id) REFERENCES $table_partners(id) ON DELETE CASCADE
    ) $charset;");

    dbDelta("CREATE TABLE $table_ip_log (
        id INT AUTO_INCREMENT PRIMARY KEY,
        partner_id INT NOT NULL,
        click_date DATE NOT NULL,
        ip_address VARCHAR(45) NOT NULL,
        UNIQUE KEY partner_date_ip (partner_id, click_date, ip_address),
        FOREIGN KEY (partner_id) REFERENCES $table_partners(id) ON DELETE CASCADE
    ) $charset;");

    if ( get_option( 'patr_limit' ) === false ) {
        add_option( 'patr_limit', 10 );
    }
}

add_action( 'wp', 'patr_track_visit_daily' );
function patr_track_visit_daily() {
    if ( is_admin() ) return;

    if ( isset( $_SERVER['HTTP_REFERER'] ) ) {
        $referer_host = parse_url( $_SERVER['HTTP_REFERER'], PHP_URL_HOST );
        if ( ! $referer_host ) return;

        global $wpdb;
        $table_partners = $wpdb->prefix . 'partner_traffic';
        $table_clicks = $wpdb->prefix . 'partner_daily_clicks';
        $table_ip_log = $wpdb->prefix . 'partner_click_ips';

        $partner = $wpdb->get_row( $wpdb->prepare( "SELECT id FROM $table_partners WHERE partner_domain = %s AND status = 'approved'", $referer_host ) );
        if ( ! $partner ) return;

        $partner_id = $partner->id;
        $today = current_time('Y-m-d');
        $user_ip = $_SERVER['REMOTE_ADDR'];

        $already_clicked = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM $table_ip_log WHERE partner_id = %d AND click_date = %s AND ip_address = %s",
            $partner_id, $today, $user_ip
        ));

        if ( $already_clicked ) return;

        $wpdb->insert( $table_ip_log, [
            'partner_id' => $partner_id,
            'click_date' => $today,
            'ip_address' => $user_ip
        ]);

        $click_entry = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_clicks WHERE partner_id = %d AND click_date = %s", $partner_id, $today ) );

        if ( $click_entry ) {
            $wpdb->query( $wpdb->prepare( "UPDATE $table_clicks SET clicks = clicks + 1 WHERE id = %d", $click_entry->id ) );
        } else {
            $wpdb->insert( $table_clicks, [
                'partner_id' => $partner_id,
                'click_date' => $today,
                'clicks' => 1
            ]);
        }
    }
}

add_shortcode( 'partner_ranking', 'patr_show_ranking_and_signup_daily' );
function patr_show_ranking_and_signup_daily( $atts ) {
    global $wpdb;
    $table_partners = $wpdb->prefix . 'partner_traffic';
    $table_clicks = $wpdb->prefix . 'partner_daily_clicks';

    $default_limit = get_option( 'patr_limit', 10 );
    $atts = shortcode_atts( ['limit' => $default_limit], $atts );
    $limit = intval( $atts['limit'] );
    if ( $limit < 1 ) $limit = 10;

    $today = current_time('Y-m-d');

    $output = '<p>' . sprintf( __( 'Top %d Partners Today', 'partner-traffic-ranker' ), $limit ) . '</p>';

    $partners = $wpdb->get_results( $wpdb->prepare(
        "SELECT p.partner_domain FROM $table_partners p
         WHERE p.status = 'approved'
         ORDER BY (
            SELECT clicks FROM $table_clicks c WHERE c.partner_id = p.id AND c.click_date = %s
         ) DESC
         LIMIT %d",
         $today, $limit
    ));

if ( $partners ) {
    $output .= '<ul>';
    foreach ( $partners as $p ) {
        $url = 'https://' . esc_attr( $p->partner_domain );
        $output .= '<li><a href="' . esc_url( $url ) . '" target="_blank" rel="dofollow">' . esc_html( $p->partner_domain ) . '</a></li>';
    }
    $output .= '</ul>';
} else {
    $output .= '<p>' . esc_html__( 'No approved partners yet.', 'partner-traffic-ranker' ) . '</p>';
}


    $message = '';
    if ( isset( $_POST['patr_submit'] ) ) {
        $domain = sanitize_text_field( $_POST['patr_domain'] );
        $domain = parse_url( esc_url_raw( $domain ), PHP_URL_HOST );

        if ( $domain ) {
            $exists = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM $table_partners WHERE partner_domain = %s", $domain ) );

            if ( $exists ) {
                $message = '<p style="color:red;">' . esc_html__( 'This domain is already registered or pending approval.', 'partner-traffic-ranker' ) . '</p>';
            } else {
                $wpdb->insert( $table_partners, [
                    'partner_domain' => $domain,
                    'status' => 'pending'
                ] );
                $message = '<p style="color:blue;">' . esc_html__( 'Partner registered! Awaiting administrator approval.', 'partner-traffic-ranker' ) . '</p>';
            }
        } else {
            $message = '<p style="color:red;">' . esc_html__( 'Please enter a valid URL.', 'partner-traffic-ranker' ) . '</p>';
        }
    }

    $output .= '<p>' . esc_html__( 'Register Your Site as a Partner (subject to approval):', 'partner-traffic-ranker' ) . '</p>';
    $output .= $message;
    $output .= '
    <form method="post">
        <p>
            <label>' . esc_html__( 'Your Site URL (e.g., https://yoursite.com):', 'partner-traffic-ranker' ) . '</label><br>
            <input type="url" name="patr_domain" required>
        </p>
        <p>
            <button type="submit" name="patr_submit">' . esc_html__( 'Register Partnership', 'partner-traffic-ranker' ) . '</button>
        </p>
    </form>';

    $output .= '<p style="margin-top:20px;">
        <a href="https://xn--flogo-dra.com/partner-auto-traffic-ranker/" target="_blank" rel="noopener">' . esc_html__( 'Download this WordPress Plugin Here', 'partner-traffic-ranker' ) . '</a>
    </p>';

    return $output;
}



// Admin Page
add_action( 'admin_menu', 'patr_settings_menu' );
function patr_settings_menu() {
    add_options_page(
        __( 'Traffic Partners', 'partner-traffic-ranker' ),
        __( 'Traffic Partners', 'partner-traffic-ranker' ),
        'manage_options',
        'ptr-partners-settings',
        'patr_admin_page'
    );
}

function patr_admin_page() {
    if ( ! current_user_can( 'manage_options' ) ) wp_die( esc_html__( 'Access denied', 'partner-traffic-ranker' ) );

    global $wpdb;
    $table_partners = $wpdb->prefix . 'partner_traffic';
    $table_clicks = $wpdb->prefix . 'partner_daily_clicks';

    if ( isset($_POST['patr_limit_submit']) && check_admin_referer('patr_limit_nonce') ) {
        $limit = intval($_POST['patr_limit']);
        if ( $limit < 1 ) $limit = 10;
        update_option( 'patr_limit', $limit );
        echo '<div class="notice notice-success"><p>' . sprintf( esc_html__( 'Display limit updated to %d.', 'partner-traffic-ranker' ), $limit ) . '</p></div>';
    }

    if ( isset($_GET['action'], $_GET['partner_id']) ) {
        $partner_id = intval($_GET['partner_id']);
        if ( $_GET['action'] === 'approve' ) {
            $wpdb->update( $table_partners, ['status' => 'approved'], ['id' => $partner_id] );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Partner approved.', 'partner-traffic-ranker' ) . '</p></div>';
        }
        if ( $_GET['action'] === 'reject' ) {
            $wpdb->update( $table_partners, ['status' => 'rejected'], ['id' => $partner_id] );
            echo '<div class="notice notice-warning"><p>' . esc_html__( 'Partner rejected.', 'partner-traffic-ranker' ) . '</p></div>';
        }
        if ( $_GET['action'] === 'delete' ) {
            $wpdb->delete( $table_partners, ['id' => $partner_id] );
            echo '<div class="notice notice-error"><p>' . esc_html__( 'Partner deleted.', 'partner-traffic-ranker' ) . '</p></div>';
        }
    }

    $today = current_time('Y-m-d');
    $partners = $wpdb->get_results( 
        "SELECT p.id, p.partner_domain, p.status, 
            (SELECT clicks FROM $table_clicks c WHERE c.partner_id = p.id AND c.click_date = '$today') AS clicks_today 
         FROM $table_partners p
         ORDER BY p.status ASC"
    );

    $limit = get_option( 'patr_limit', 10 );

    ?>
    <div class="wrap">
        <p><?php esc_html_e( 'Traffic Partners', 'partner-traffic-ranker' ); ?></p>

        <form method="post" style="margin-bottom:30px;">
            <?php wp_nonce_field('patr_limit_nonce'); ?>
            <label for="patr_limit"><strong><?php esc_html_e( 'Number of partners to show in ranking:', 'partner-traffic-ranker' ); ?></strong></label>
            <input type="number" min="1" id="patr_limit" name="patr_limit" value="<?php echo esc_attr( $limit ); ?>" style="width:60px; margin-left:10px;">
            <input type="submit" name="patr_limit_submit" class="button button-primary" value="<?php esc_attr_e( 'Save', 'partner-traffic-ranker' ); ?>">
        </form>

        <table class="widefat">
            <thead>
                <tr>
                    <th><?php esc_html_e( 'Domain', 'partner-traffic-ranker' ); ?></th>
                    <th><?php esc_html_e( 'Clicks Today', 'partner-traffic-ranker' ); ?></th>
                    <th><?php esc_html_e( 'Status', 'partner-traffic-ranker' ); ?></th>
                    <th><?php esc_html_e( 'Actions', 'partner-traffic-ranker' ); ?></th>
                </tr>
            </thead>
            <tbody>
            <?php if ( $partners ) :
                foreach ( $partners as $p ) : ?>
                <tr>
                    <td><?php echo esc_html( $p->partner_domain ); ?></td>
                    <td><?php echo intval( $p->clicks_today ); ?></td>
                    <td><?php echo esc_html( ucfirst($p->status) ); ?></td>
                    <td>
                        <?php if ( $p->status !== 'approved' ) : ?>
                            <a href="<?php echo admin_url( 'options-general.php?page=ptr-partners-settings&action=approve&partner_id=' . $p->id ); ?>" class="button"><?php esc_html_e( 'Approve', 'partner-traffic-ranker' ); ?></a>
                        <?php endif; ?>
                        <?php if ( $p->status !== 'rejected' ) : ?>
                            <a href="<?php echo admin_url( 'options-general.php?page=ptr-partners-settings&action=reject&partner_id=' . $p->id ); ?>" class="button"><?php esc_html_e( 'Reject', 'partner-traffic-ranker' ); ?></a>
                        <?php endif; ?>
                        <a href="<?php echo admin_url( 'options-general.php?page=ptr-partners-settings&action=delete&partner_id=' . $p->id ); ?>" class="button" onclick="return confirm('<?php echo esc_js( __( 'Are you sure you want to delete?', 'partner-traffic-ranker' ) ); ?>');"><?php esc_html_e( 'Delete', 'partner-traffic-ranker' ); ?></a>
                    </td>
                </tr>
                <?php endforeach; 
            else: ?>
                <tr><td colspan="4"><?php esc_html_e( 'No partners registered.', 'partner-traffic-ranker' ); ?></td></tr>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
    <?php
}

// Widget
class Partner_Ranking_Widget extends WP_Widget {
    public function __construct() {
        parent::__construct(
            'partner_ranking_widget',
            __( 'Daily Partner Ranking', 'partner-traffic-ranker' ),
            ['description' => __( 'Shows the partner sites sending most visits today, daily reset at midnight.', 'partner-traffic-ranker' )]
        );
    }

    public function widget( $args, $instance ) {
        echo $args['before_widget'];

        $limit = get_option( 'patr_limit', 10 );

        if ( ! empty( $instance['title'] ) ) {
            echo $args['before_title'] . apply_filters( 'widget_title', $instance['title'] ) . $args['after_title'];
        } else {
            echo $args['before_title'] . sprintf( __( 'Top %d Partners', 'partner-traffic-ranker' ), $limit ) . $args['after_title'];
        }

        echo do_shortcode('[partner_ranking limit=' . $limit . ']');

        echo $args['after_widget'];
    }

    public function form( $instance ) {
        $title = ! empty( $instance['title'] ) ? $instance['title'] : __( 'Top Partners', 'partner-traffic-ranker' );
        ?>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id('title') ); ?>"><?php esc_html_e( 'Title:', 'partner-traffic-ranker' ); ?></label> 
            <input class="widefat" id="<?php echo esc_attr( $this->get_field_id('title') ); ?>" 
                   name="<?php echo esc_attr( $this->get_field_name('title') ); ?>" type="text" 
                   value="<?php echo esc_attr( $title ); ?>">
        </p>
        <?php 
    }

    public function update( $new_instance, $old_instance ) {
        $instance = [];
        $instance['title'] = sanitize_text_field( $new_instance['title'] );
        return $instance;
    }
}
add_action( 'widgets_init', function() {
    register_widget( 'Partner_Ranking_Widget' );
});
